///////
     //
     //  MOBOTIX EventStream Client SDK 
     //
     //  This software is licensed under the BSD license,
     //  http://www.opensource.org/licenses/bsd-license.php
     //
     //  Copyright (c) 2005 - 2016, MOBOTIX AG
     //  All rights reserved.
     //
     //  Redistribution and use in source and binary forms, with or without
     //  modification, are permitted provided that the following conditions are
     //  met:
     //
     //  - Redistributions of source code must retain the above copyright
     //    notice, this list of conditions and the following disclaimer.
     //
     //  - Redistributions in binary form must reproduce the above copyright
     //    notice, this list of conditions and the following disclaimer in the
     //    documentation and/or other materials provided with the distribution.
     //
     //  - Neither the name of MOBOTIX AG nor the names of its contributors may
     //    be used to endorse or promote products derived from this software
     //    without specific prior written permission.
     //
     //  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
     //  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
     //  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
     //  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
     //  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
     //  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
     //  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
     //  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
     //  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
     //  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
     //  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
     //
///////

/*
 *
 * This is a simple example application that shows how to fetch live video and audio from the camera. The received
 * stream is decoded and stored as raw data in separate files for video and audio.
 *
 * To play those raw files you can use vlc:
 *
 * Audio:
 *
 * vlc --demux=rawaud --rawaud-channels 1 --rawaud-samplerate 16000 audio_stream.raw
 *
 * Video:
 *
 * vlc --demux rawvideo --rawvid-fps 12 --rawvid-width <WIDTH> --rawvid-height <HEIGHT> --rawvid-chroma I420 video_stream.raw
 *
 */

#include "MxPEG_SDK_API.hpp"
#include "nlohmann/json.hpp"

#include "SinkVideo.h"
#include "SinkAudio.h"
#include "AuthorizationHandler.h"
#include "NotificationListener.h"

#include <iostream>
#include <memory>
#include <string>
#include <chrono>

#ifndef WINDOWS
#include <sys/time.h>
#include <unistd.h>
#endif
#define URL_SZ 256

using namespace ie::MxPEG;

/*
 * Exporting content requires the client to keep track of certain states while it issues the required
 * commands. This class is a simple example that shows how to implement that.
 *
 * The camera processes all commands asynchronously. If we simply issue the commands without waiting for their confirmation, unexpected behavior might occur
 * in this scenario.For example if the export is started before the seek command has completed, frames
 * outside the requested range might be sent by the camera. Therefore we need to wait for confirmation at certain points of the process before we can
 * continue sending commands.
 *
 * The export requires a couple of steps of which each needs to be completed before the next one can be started:
 *
 * 1. Setup the export
 *
 * 2. Issue a seek to the required start position
 *
 * 3. Start the export and check for the end condition
 *
 * 4. Do a little cleanup
 *
 * Please note: This example does not include any timeout mechanisms! If a camera becomes unavailable while an instance of the ExportManager is waiting
 * for a response, the according while loop may nerver terminate. In a production system adding a timeout mechanism in the different while loops is recommended.
 */
class ExportManager {
public:
   ExportManager(MxPEG_SDK::shared_ptr_t client, MxPEG_SinkVideo::shared_ptr_t sinkVideo, NotificationListener::shared_ptr_t notificationListener);

   bool startExport(struct timeval startTime, struct timeval stopTime, double rate);

private:
   struct timeval m_startTime;
   struct timeval m_stopTime;
   MxPEG_SDK::shared_ptr_t m_client;
   MxPEG_SinkVideo::shared_ptr_t m_sinkVideo;
   NotificationListener::shared_ptr_t m_notificationListener;
};

ExportManager::ExportManager(MxPEG_SDK::shared_ptr_t client, MxPEG_SinkVideo::shared_ptr_t sinkVideo, NotificationListener::shared_ptr_t notificationListener)
:m_client(client), m_sinkVideo(sinkVideo), m_notificationListener(notificationListener)
{
   m_startTime.tv_sec=0;
   m_startTime.tv_usec=0;
   m_stopTime.tv_sec=0;
   m_stopTime.tv_usec=0;
}

bool ExportManager::startExport(struct timeval startTime, struct timeval stopTime, double rate) {
   m_startTime = startTime;
   m_stopTime = stopTime;

   /*
    * Activate export mode for this session
    */
   int exp_id = m_client->setExportMode(true);

   /*
    * Set the desired speed for the export. If speed factor is exceeds the maximum possible speed, the camera will send at maximum speed.
    */
   int rate_id = m_client->setPlaybackRate(rate);

   /*
    * Now call the clients loop() function until we have confirmation for the queued commands
    */
   while(exp_id > 0 && rate_id > 0) {
      std::string tmp;
      if(exp_id > 0 && m_notificationListener->haveResult(exp_id, tmp)) {
             std::cout << " Activated export mode: " << tmp << std::endl;
             exp_id = -1;
      }
      if(rate_id > 0 && m_notificationListener->haveResult(rate_id, tmp)) {
             std::cout << " Set playback rate to " << rate << ": " << tmp << std::endl;
             rate_id = -1;
      }
      m_client->loop();
   }


   /*
    * Queue the seek command to the specified start time
    */
   int seek_id = m_client->seek(MxPEG_SeekMode::sm_previous, false, m_startTime);
   /*
    * Call loop again until we get a seek reply - check the reply for errors in case seek fails!
    *
    * Note: The seek operation can take quite long to finish - Especially if the camera uses a network storage with
    * gigabytes of footage (in such a case it seek can take minutes to complete!).
    */
   while(true) {
      std::string seekResult;
      if(m_notificationListener->haveResult(seek_id, seekResult)) {
         //to process the notification response we need to parse the json structure:
         nlohmann::json j_message = nlohmann::json::parse(seekResult);
         //check if we have an error:
         if(j_message["error"].is_null()) {
            //no error: JSON looks like this: {"result":["pause",2478,"1350909067.184786"],"error":null,"id":22}
            std::cout << "Successfully seeked to " << j_message["result"].at(2) << std::endl;
            break;
         } else {
            //we have an error element like this: {"result":null,"error":[0,"invalid or missing parameter"],"id":11}
            std::cout << "Seek failed with  " << j_message["error"].at(1) << std::endl;
            return false;
         }
      }
      m_client->loop();
   }

   //drop stored results at this point - we don't need them in this example anymore
   m_notificationListener->clearResults();

   /*
    * Queue the play command to start the export
    */
   int play_id = m_client->play();
   /*
    * Call loop until we reach the specified stop time or we get a reply to the play command
    */
   while(true) {
      std::string playResult;
      if(m_notificationListener->haveResult(play_id, playResult)) {
         //to process notification response we need to parse the json structure:
         nlohmann::json j_message = nlohmann::json::parse(playResult);
         if(j_message["error"].is_null()) {
            //no error: JSON looks like this: {"result":["play",1,"1653038936.173995"],"error":null,"id":12}
            std::cout << "Successfully started export at " << j_message["result"].at(2) << std::endl;
            //we continue this loop until we get an end of stream or reach the stop time
            //drop stored results at this point to avoid repeating this message
            m_notificationListener->clearResults();
         } else {
            nlohmann::json j_error_id = j_message["error"].at(0);
            //check if we got an "end of stream" -> not a real error
            if(j_error_id.is_number() && j_error_id.get<int>() == 37) {
               std::cout << "Reached end of recorded content - stop" << std::endl;
               break;
            } else {
               //we have an error element like this: {"result":null,"error":[0,"invalid or missing parameter"],"id":11}
               std::cout << "Play/Export failed with  " << j_message["error"].at(1) << std::endl;
               return false;
            }
         }
      }

      //check if we reached the end of the requested interval
      struct timeval currentFrame = std::static_pointer_cast<SinkVideo>(m_sinkVideo)->lastFrameTime();
      if(m_stopTime.tv_sec > 0 && //make sure a stop time is set
            currentFrame.tv_sec >= stopTime.tv_sec && currentFrame.tv_usec >= stopTime.tv_usec) {
         std::cout << "Reached end of requested interval - stop" << std::endl;
         break;
      }

      m_client->loop();
   }

   //drop stored results at this point - we don't need it in this example anymore
   m_notificationListener->clearResults();

   std::cout << " Export completed: Stopping stream and ending export mode" << std::endl;


   /*
    * Send pause and wait until the stream has stopped
    */
   play_id = m_client->pause();

   while(true) {
      std::string tmp;
      if(play_id > 0 && m_notificationListener->haveResult(play_id, tmp)) {
             std::cout << " Stopped export: " << tmp << std::endl;
             break;
      }
      m_client->loop();
   }

   /*
    * disable export mode and reset playback rate to default
    * 1st Queue commands
    * 2nd Call loop until we have confirmation for both commands
    */
   exp_id = m_client->setExportMode(false);
   rate_id = m_client->setPlaybackRate(1.0);

   while(exp_id > 0 && rate_id > 0) {
      std::string tmp;
      if(exp_id > 0 && m_notificationListener->haveResult(exp_id, tmp)) {
             std::cout << " Disabled export: " << tmp << std::endl;
             exp_id = -1;
      }
      if(rate_id > 0 && m_notificationListener->haveResult(rate_id, tmp)) {
             std::cout << " Reset playback rate to 1.0: " << tmp << std::endl;
             rate_id = -1;
      }
      m_client->loop();
   }

   /*
    * Drop all queued notifications in listener (we don't need them in this example anymore)
    */
   m_notificationListener->clearResults();

   return true;
}

int main(int argc, char** argv) {
    std::cout << "EventStream client SDK live stream example" << std::endl;

    if(argc < 4) {
        std::cout << "Missing parameter Hostname/IP" << std::endl;
        std::cout << "Start with " << argv[0] << " http|https <hostname|IP> <startTime> [<stopTime>]" << std::endl;
        std::cout << "<startTime>: start time as unix time in seconds since epoch" << std::endl;
        std::cout << "[<startTime>]: (optional) stop time as unix time in seconds since epoch" << std::endl;
        std::cout << "Press enter to exit" << std::endl;
        getchar();
        exit(1);
    }

    char url[URL_SZ] = "";
    snprintf(url,URL_SZ,"%s://%s/control/eventstream.jpg", argv[1], argv[2]);

#ifdef _MSC_VER
    WSADATA wsaData;
    int iResult;
    iResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
    if (iResult != NO_ERROR) {
        std::cout << "WSAStartup failed: " << iResult << std::endl;
        return 1;
    }
#endif

    struct timeval startTime;
    startTime.tv_sec = 0;
    startTime.tv_usec = 0;
    startTime.tv_sec = (long int)std::stol(argv[3], nullptr, 10);

    struct timeval stopTime;
    stopTime.tv_sec = 0;
    stopTime.tv_usec = 0;
    if(argc >= 5) {
       stopTime.tv_sec = (long int)std::stol(argv[4], nullptr, 10);
    }

    std::cout << "Exporting content from " << startTime.tv_sec << " until ";
    if(stopTime.tv_sec > 0) {
       std::cout << stopTime.tv_sec << std::endl;
    } else {
       std::cout << "end of recordings (end of stream)" << std::endl;
    }

    /*
    * Create a video sink object to process the decoded video frames
    */
    MxPEG_SinkVideo::shared_ptr_t sinkVideo = MxPEG_SinkVideo::shared_ptr_t(new SinkVideo("video_stream"));

    /*
    * Create an audio sink object to process the received audio packages
    */
    MxPEG_SinkAudio::shared_ptr_t sinkAudio = MxPEG_SinkAudio::shared_ptr_t(new SinkAudio("audio_stream"));

    /*
    * Create an authorization handler that provides the credentials (if needed)
    */
    MxPEG_AuthorizationHandler::shared_ptr_t authHandler = MxPEG_AuthorizationHandler::shared_ptr_t(new AuthorizationHandler());

    /*
     * Create a notification listener to handle the json replys of the camera
     * Note: We use the shared_ptr type defined for the object type of this sample app, so tha we can access the extra API 
    * functions from the main loop further down here.
    */
    NotificationListener::shared_ptr_t notificationListener = NotificationListener::shared_ptr_t(new NotificationListener());

    /*
    * Create a new eventstream client object. We use use decoding mode MxPEG (= no decoding)
    *
    * This means the MxPEG frames are passed as they are received to the video sink. This way we can avoid the CPU
    * load required for decoding the image stream in this sample application. Metadata and ThermalRAW data is extracted
    * and available in the sink objects as usual.
    */
    MxPEG_SDK::shared_ptr_t client = MxPEG_SDK_Factory::create(sinkAudio, sinkVideo, MxPEG_ImageMode::im_MxPEG, authHandler,notificationListener);

    //client->set_log_mask(0);

    /*
    * configure the client to connect to camera with hostname "i25"
    */
    if (client->stream_setup(url) < 0) {
         std::cerr << __PRETTY_FUNCTION__ << ": Couldn't setup stream from " << url << std::endl;
    } else {
        /*
        * activate audio for this session (only has an effect if the camera has a microphone that is activated)
        */
        client->setAudioActive(true);

        /*
        * ativate video for this session
        */
        client->setVideoActive(true);

        /*
         * activate raw thermal data export for this session
         *
         * -> Video export needs to be activated as well, otherwise no thermal raw data weill be sent by the camera
         *
         * -> To use this feature the camera must be equipped with at least one thermal sensor that is configured
         * for raw export. (Open the camera setup menu and navigate to: Setup->Thermal Sensor Settings, set the option
         * "Thermal Raw Data" to "enabled".
         *
         * See thermal-raw sample for more details how to process thermal data
         */
        client->setThermalRawExportActive(true);

        ExportManager exportManager(client, sinkVideo, notificationListener);

        /*
         * The export manager will wrap all tasks needed to export the requested stream to the sink classes.
         */
        exportManager.startExport(startTime, stopTime, 20.0);

#if 0
        /*
         * activate export mode for this session
         */
        client->setExportMode(true);

        /*
         * set the desired speed for the export. If speed factor is exceeds the maximum possible speed, the camera will send at maximum speed.
         */
        client->setPlaybackRate(20.0);

        /*
         * seek to the specified start time
         */
        int seek_id = client->seek(MxPEG_SeekMode::sm_previous, false, startTime);
        notificationListener->setSeekTaskId(seek_id);

        /*
        * id that will be set once the play command has been sent
        */
        int export_id = -1;
         


        struct timeval lastUpdate;
        lastUpdate.tv_sec = 0;
        lastUpdate.tv_usec = 0;
        struct timeval lastFrame;
        lastFrame.tv_sec = 0;
        lastFrame.tv_usec = 0;

        /*
        * The SDK library is single threaded, the function loop() acts as the main loop of the library. Call it repeatedly
        * from the applications main loop.
        */
        while(true) {
           /*
            * call loop to process IO
            */
           client->loop();

           /*
            * If an end time has been set, check the timestamp of the current frame to see if the end time has been reached.
            */
           struct timeval currentFrame = std::static_pointer_cast<SinkVideo>(sinkVideo)->lastFrameTime();
           if(stopTime.tv_sec > 0 && //make sure a stop time is set
                 currentFrame.tv_sec >= stopTime.tv_sec && currentFrame.tv_usec >= stopTime.tv_usec) {
              std::cerr << __PRETTY_FUNCTION__ << ": Reached end of requested interval - exit" << std::endl;
              break;
           }

            /* 
             * Check if seek was successful, then issue the play command (once)
             */
           if(notificationListener->isSeekSuccess() && export_id == -1) {
                /*
                * start the expor, store the export task id so that we can track responses from the camera in our Notification listener instance
                */
                export_id = client->play();

                /*
                * Now pass the ID of the export task to the notification listener. This way the notification listener can watch 
                * for events related to the export
                */
                std::cerr << __PRETTY_FUNCTION__ << "Setting export id " << export_id << std::endl;
                notificationListener->setExportTaskId(export_id);
           }

            /* 
             * Check if the notification listener has received an end of stream event (export finished)
             */
           if(notificationListener->isExportFinished()) {
              std::cerr << __PRETTY_FUNCTION__ << ": Export completed - got end of stream message - exit" << std::endl;
              std::cerr << __PRETTY_FUNCTION__ << ":" << notificationListener->getExportResultMsg() << std::endl;
              break;
           }

           /*
            * Please note: This is a fallback in case the end of stream event is missing
            *
            * Check if we still get new frames
            * - the video stream will never start if we have no recordings after the start time
            * - the video stream will simply stop if we reach the end of available recordings
            *
            * A better solution would be to check for the "end of stream" message in the NotificationListener:
            *     {"result":null,"error":[37,"end of stream"],"id":11}
            * The "id" of that message will match the id returned to the "client->play();" invocation.
            */
           struct timeval now;
           gettimeofday(&now, NULL);
           if(lastFrame.tv_sec > 0 &&
                 currentFrame.tv_sec == lastFrame.tv_sec && currentFrame.tv_usec == lastFrame.tv_usec ) {
              /* We have no new frame, check for timeout (granularity seconds)
               *
               * Attention: If there is a large number of recordings, the seek (to the start position) might take longer
               * that the 5 second timeout used here.
               */
              int64_t diff = now.tv_sec - lastUpdate.tv_sec;
              if(diff > 5) {
                 std::cerr << __PRETTY_FUNCTION__ << ": No new frames for 5 seconds - assuming end of export - exit" << std::endl;
                 break;
              }
           } else {
              lastUpdate = now;
              lastFrame = currentFrame;
           }

        }
#endif
    }

#ifdef _MSC_VER
    WSACleanup();
#endif
    return 0;
}
