///////
     //
     //  MOBOTIX EventStream Client SDK 
     //
     //  This software is licensed under the BSD license,
     //  http://www.opensource.org/licenses/bsd-license.php
     //
     //  Copyright (c) 2005 - 2016, MOBOTIX AG
     //  All rights reserved.
     //
     //  Redistribution and use in source and binary forms, with or without
     //  modification, are permitted provided that the following conditions are
     //  met:
     //
     //  - Redistributions of source code must retain the above copyright
     //    notice, this list of conditions and the following disclaimer.
     //
     //  - Redistributions in binary form must reproduce the above copyright
     //    notice, this list of conditions and the following disclaimer in the
     //    documentation and/or other materials provided with the distribution.
     //
     //  - Neither the name of MOBOTIX AG nor the names of its contributors may
     //    be used to endorse or promote products derived from this software
     //    without specific prior written permission.
     //
     //  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
     //  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
     //  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
     //  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
     //  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
     //  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
     //  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
     //  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
     //  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
     //  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
     //  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
     //
///////
#include <iostream>
#include <cstdio>
#include <sstream>

#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#if defined (_MSC_VER)
#include <fcntl.h>
#include <io.h>
#else
#include <unistd.h>
#endif

#include <filesystem>
#include <iostream>
#include <fstream>
#include <ostream>
#include <vector>
#include <cstdint>
#include <cstring>
#include <sstream>

#include "mxpeg-decoder.h"


bool write_image_data(MxPEG_Image & buffer, const std::string & name) {

    char fname[1024];
    switch (buffer.mode()) {
    case MxPEG_ImageMode::im_RGB:
        snprintf(fname, 1024, "%s_%ux%u.rgb", name.c_str(), buffer.width(), buffer.height());
        break;
    case MxPEG_ImageMode::im_YUV:
        snprintf(fname, 1024, "%s_%ux%u.yuv", name.c_str(), buffer.width(), buffer.height());
        break;
    default:
        snprintf(fname, 1024, "%s_%ux%u.yuvx", name.c_str(), buffer.width(), buffer.height());
        break;
    }

    ofstream out_file(fname, ifstream::binary);
    if (!out_file.is_open())
    {
        return false;
    }


    //only write RGB data
    if ( buffer.mode() == MxPEG_ImageMode::im_RGB )
    {
        /*
        * channel order is bgra, interleaved 4:4:4, 8 bit per channel
        */
        const uint8_t *rgbBuffer = buffer.imgBuffer();
        const size_t rgbBufferSize = buffer.imgBufferSize();
        out_file.write(reinterpret_cast<const char *>(rgbBuffer), rgbBufferSize);
        //fwrite(rgbBuffer, rgbBufferSize, 1, fVideoOut);
        std::cout << " -> wrote raw rgb image: " << rgbBufferSize << " bytes, "
                  << " res: " << (int)buffer.width() << "x"
                  << (int)buffer.height() << std::endl;
    } else {
        //yuv 420 planar
        const uint8_t * yBuffer = buffer.imgBuffer();
        uint32_t  yBufferSize = buffer.width() * buffer.height();

        const uint8_t * uBuffer = yBuffer + yBufferSize;
        uint32_t  uBufferSize = buffer.width()/2 * buffer.height()/2;

        const uint8_t * vBuffer = uBuffer + uBufferSize;
        uint32_t  vBufferSize = buffer.width()/2 * buffer.height()/2;

        out_file.write(reinterpret_cast<const char *>(yBuffer), yBufferSize);
        out_file.write(reinterpret_cast<const char *>(uBuffer), uBufferSize);
        out_file.write(reinterpret_cast<const char *>(vBuffer), vBufferSize);
    }

    out_file.close();
    return true;
}

bool write_thermal_raw(std::shared_ptr<MX_ThermalRawData> rawData,  const std::string & name) {

    char fname[1024];
    snprintf(fname, 1024, "%s_%s_%ux%u_%s.thermal.raw",
             name.c_str(),
             ((rawData->sensor()== MXT_Sensor::left)? "left" : "right"),
             rawData->width(),
             rawData->height(),
             ((rawData->bitDepth()== MXT_BitDepth::depth14bit)? "14bit" : "unknown" ));

    ofstream out_file(fname, ifstream::binary);
    if (!out_file.is_open()) {
        return false;
    }


    /*
    * channel order is bgra, interleaved 4:4:4, 8 bit per channel
    */
    const uint8_t * rawBuffer = rawData->rawBuffer();
    size_t rawBufferSize = rawData->rawBufferSize();

    out_file.write(reinterpret_cast<const char *>(rawBuffer), rawBufferSize);
    std::cout << " -> wrote thermal raw data of sensor " << (int)rawData->sensor()
              << " " << rawBufferSize << " bytes, " << " res: " << (int)rawData->width() << "x"
              << (int)rawData->height() << std::endl;

    out_file.close();
    return true;
}

bool write_thermal_raw_int_csv(std::shared_ptr<MX_ThermalRawData> rawData, const std::string & name)
{
    char fname[1024];
    snprintf(fname, 1024, "%s_%s_%ux%u_%s.thermal.uint.csv",
             name.c_str(),
             ((rawData->sensor()== MXT_Sensor::left)? "left" : "right"),
             rawData->width(),
             rawData->height(),
             ((rawData->bitDepth()== MXT_BitDepth::depth14bit)? "14bit" : "unknown"));


    ofstream out_file(fname, ifstream::binary);
    if (!out_file.is_open()) {
        return false;
    }

    std::ostringstream header;
    header << "sensor;" << ((rawData->sensor()== MXT_Sensor::left)? "left" : "right") << "\n";
    header << "bit depth;" << ((rawData->bitDepth()== MXT_BitDepth::depth14bit)? "14 bit" : "unknown") << "\n";
    header << "width;" << (int)rawData->width() << "\n";
    header << "height;" << (int)rawData->height() << "\n";
    header << "resolution mK;" << (int)rawData->resolution_mK() << "\n";
    header << "advanced radiometry support;" << ((rawData->advancedRadiometrySupport()==true)? "yes" :" no") << "\n";
    header << "unit;raw uint16" << "\n";
    header << "\n";

    out_file.write(header.str().c_str(), header.str().length());

    /*
    * raw layout of 14bit thermal raw:
    * format uses 2 bytes per pixel (little endian), with the 2 most significant bytes set to 0:
    * llll llll hhhh hhhh
    * xxxx xxxx 00xx xxxx
    */
    const uint8_t * rawBuffer = rawData->rawBuffer();
    size_t rawBufferSize = rawData->rawBufferSize();
    for(uint32_t y = 0; y < rawData->height(); ++y) {
        std::ostringstream line;
        for(uint32_t x = 0; x < rawData->width(); ++x) {
            uint32_t offset = y*(rawData->width()*2) + (x * 2);
            //sanity check - should never happen
            if((offset + 1) >= rawBufferSize) break;
            //make sure to clear 2 msb
            uint8_t highByte = (rawBuffer[offset+1] & 0x3F);
            uint8_t lowByte = rawBuffer[offset];
            uint32_t value = (highByte << 8) + lowByte;
            line << value;
            if(x+1 < rawData->width()) line << ";";
        }
        line << "\n";
        out_file.write(line.str().c_str(), line.str().length());
    }

    std::cout << "  -> converted thermal raw data of sensor " << (int)rawData->sensor() << " to integer csv file " << std::endl;

    out_file.close();
    return true;
}


bool write_thermal_celcius_csv(std::shared_ptr<MX_ThermalRawData> rawData,  const std::string & name) {

    if(!rawData->advancedRadiometrySupport()) {
        std::cout << "conversion to absolute temperatures is only supported for sensors with advanced radiometry support" << std::endl;
        return false;
    }

    char fname[1024];
    snprintf(fname,1024,"%s_%s_%ux%u_%s.thermal.clecius.csv",
             name.c_str(),
             ((rawData->sensor()== MXT_Sensor::left)? "left" : "right"),
             rawData->width(),
             rawData->height(),
             ((rawData->bitDepth()== MXT_BitDepth::depth14bit)? "14bit" : "unknown"));


    ofstream out_file(fname, ifstream::binary);
    if (!out_file.is_open()) {
        return false;
    }


    std::ostringstream header;
    header << "sensor;" << ((rawData->sensor()== MXT_Sensor::left)? "left" : "right") << "\n";
    header << "bit depth;" << ((rawData->bitDepth()== MXT_BitDepth::depth14bit)? "14 bit" : "unknown") << "\n";
    header << "width;" << (int)rawData->width() << "\n";
    header << "height;" << (int)rawData->height() << "\n";
    header << "resolution mK;" << (int)rawData->resolution_mK() << "\n";
    header << "advanced radiometry support;" << ((rawData->advancedRadiometrySupport()==true)?" yes" : "no") << "\n";
    header << "unit;degrees Celsius" << "\n";

    header << "\n";

    out_file.write(header.str().c_str(), header.str().length());

    for(uint32_t y = 0; y < rawData->height(); ++y) {
        std::ostringstream line;
        for(uint32_t x = 0; x < rawData->width(); ++x) {
            line << rawData->convertTo(x,y,MXT_Unit::celsius);
            if(x+1 < rawData->width()) line << ";";
        }
        line << "\n";
        out_file.write(line.str().c_str(), line.str().length());
    }

    std::cout << "  -> converted thermal raw data of sensor " << (int)rawData->sensor() << " to celcius csv file " << std::endl;

    out_file.close();
    return true;

}


void write_thermal_data(MxPEG_Image &buffer, const std::string &name) {

    std::shared_ptr<MX_ThermalRawData> rawData = buffer.fetchThermalRawData(MXT_Sensor::left);
    if(rawData.get() != nullptr) {
        std::cout << "  Got thermal raw data from left sensor " << std::endl;
        write_thermal_raw(rawData, name);
        write_thermal_raw_int_csv(rawData, name);
        write_thermal_celcius_csv(rawData, name);
    }

    rawData = buffer.fetchThermalRawData(MXT_Sensor::right);
    if(rawData.get() != nullptr) {
        std::cout << "  Got thermal raw data from right sensor " << std::endl;
        write_thermal_raw(rawData, name);
        write_thermal_raw_int_csv(rawData, name);
        write_thermal_celcius_csv(rawData, name);
    }
}


bool  decode_frame(MxPEG_Decoder_API::shared_ptr_t decoder, uint8_t *data, size_t data_size, const std::string  &out_frame_decoded) {

    /*
    * Setup data structure for decoder
    */
    MxPEG_Frame frame;
    frame.data = data;
    frame.size = data_size;

    /*
    * Decode the frame, the decoded image will be similar to what is passed to the VideoSink classes in the streaming use case
    */
    MxPEG_Image::unique_ptr_t decodedImage = decoder->decodeFrame(frame);

    /*
    * Write the decoded raw data to output files
    */
    if (decodedImage) {
        write_image_data(*decodedImage, out_frame_decoded);
        write_thermal_data(*decodedImage, out_frame_decoded);
    } else {
        std::cout << "No image decoded." << std::endl;
    }
    return  true;
}

